import unittest
import os
from time import strftime
from ..mpr import mpr_scp, mpr_applyrules, mpr_getsshkey, mpr_sshkey


class TestMpr(unittest.TestCase):
    def test_mpr_ssh(self):
        timestamp = strftime('%Y%m%d%H%M%S_%s')
        key_name = '/tmp/mpr_ssh_test_%s' % timestamp
        pubkey_name = key_name+'.pub'
        key = mpr_getsshkey(keyloc=pubkey_name)

        self.assertFalse(key)

        mpr_sshkey(pkeyloc=key_name, overwrite=True)
        key = mpr_getsshkey(keyloc=pubkey_name)

        if os.path.isfile(key_name):
            os.remove(key_name)
        if os.path.isfile(pubkey_name):
            os.remove(pubkey_name)

        self.assertIsNotNone(key)

    def test_mpr_scp(self):
        # Dryrun with sudo
        res = mpr_scp(
            ip='192.168.10.11',
            file=b'Dies kommt in die Datei',
            location='/etc/openvpn/up-cmd-tun',
            proxyhost='vpnnode1',
            proxyuser='mpaproxy',
            identity='/home/mpaproxy/.ssh/mpr-id_rsa',
            dryrun=True
        )
        expected = {
            'status': 0,
            'error': 0,
            'cmd': '',
            'msg': (
                'scp /tmp/dryrun_file mpaproxy@vpnnode1:/tmp/dryrun_file2\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'scp -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa /tmp/dryrun_file2 '
                'maint@192.168.10.11:/tmp/dryrun_file3\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa maint@192.168.10.11 '
                'sudo mv /tmp/dryrun_file3 /etc/openvpn/up-cmd-tun'
            )
        }
        self.assertTrue(res == expected, msg=res)

        # Dryrun with sudo and chown
        res = mpr_scp(
            ip='192.168.10.11',
            file=b'Dies kommt in die Datei',
            location='/etc/openvpn/up-cmd-tun',
            proxyhost='vpnnode1',
            proxyuser='mpaproxy',
            identity='/home/mpaproxy/.ssh/mpr-id_rsa',
            chown='perfact',
            dryrun=True
        )
        expected = {
            'status': 0,
            'error': 0,
            'cmd': '',
            'msg': (
                'scp /tmp/dryrun_file mpaproxy@vpnnode1:/tmp/dryrun_file2\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'scp -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa /tmp/dryrun_file2 '
                'maint@192.168.10.11:/tmp/dryrun_file3\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa maint@192.168.10.11 '
                'sudo mv /tmp/dryrun_file3 /etc/openvpn/up-cmd-tun\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa maint@192.168.10.11 '
                'sudo chown perfact:perfact /etc/openvpn/up-cmd-tun'
            )
        }
        self.assertTrue(res == expected, msg=res)

        # Dryrun with sudo and chown and chmod
        res = mpr_scp(
            ip='192.168.10.11',
            file=b'Dies kommt in die Datei',
            location='/etc/openvpn/up-cmd-tun',
            proxyhost='vpnnode1',
            proxyuser='mpaproxy',
            identity='/home/mpaproxy/.ssh/mpr-id_rsa',
            chmod='777',
            chown='perfact',
            dryrun=True
        )
        expected = {
            'status': 0,
            'error': 0,
            'cmd': '',
            'msg': (
                'scp /tmp/dryrun_file mpaproxy@vpnnode1:/tmp/dryrun_file2\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'scp -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa /tmp/dryrun_file2 '
                'maint@192.168.10.11:/tmp/dryrun_file3\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa maint@192.168.10.11 '
                'sudo mv /tmp/dryrun_file3 /etc/openvpn/up-cmd-tun\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa maint@192.168.10.11 '
                'sudo chown perfact:perfact /etc/openvpn/up-cmd-tun\n'
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa maint@192.168.10.11 '
                'sudo chmod 777 /etc/openvpn/up-cmd-tun'
            )
        }
        self.assertTrue(res == expected, msg=res)

        # Dryrun with dangerous bash character
        res = mpr_scp(
            ip='192.168.10.11',
            file=b'Dies kommt in die Datei',
            location='/etc/openvpn/up-cmd-tun || dangerous',
            proxyhost='vpnnode1',
            proxyuser='mpaproxy',
            identity='/home/mpaproxy/.ssh/mpr-id_rsa',
            dryrun=True
        )
        expected = {
            'status': 2,
            'error': 0,
            'cmd': '',
            'msg': (
                'Forbidden characters for bash found in: '
                '/etc/openvpn/up-cmd-tun || dangerous'
            ),
        }
        self.assertTrue(res == expected, msg=res)

        # Dryrun and only specify the target directory and not a path
        res = mpr_scp(
            ip='192.168.10.11',
            file=b'Dies kommt in die Datei',
            location='/etc/openvpn/',
            proxyhost='vpnnode1',
            proxyuser='mpaproxy',
            identity='/home/mpaproxy/.ssh/mpr-id_rsa',
            dryrun=True
        )
        expected = {
            'status': 1,
            'error': 0,
            'cmd': '',
            'msg': (
                'Please specify the full path and not just the directory'
            ),
        }
        self.assertTrue(res == expected, msg=res)

    def test_mpr_applyrules(self):
        # Test basic funtionality
        res = mpr_applyrules(
            ip='192.168.10.11',
            proxyhost='vpnnode1',
            proxyuser='mpaproxy',
            identity='/home/mpaproxy/.ssh/mpr-id_rsa',
            dryrun=True
        )
        expected = {
            'status': 0,
            'error': 0,
            'cmd': '',
            'msg': (
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa maint@192.168.10.11 '
                '"sudo bash /etc/openvpn/down-cmd-tun && '
                'sudo bash /etc/openvpn/up-cmd-tun && '
                'sudo mv /etc/openvpn/down-cmd-tun-new '
                '/etc/openvpn/down-cmd-tun"'
            )
        }
        self.assertTrue(res == expected, msg=res)

        # Test with some different default values
        res = mpr_applyrules(
            ip='192.168.10.11',
            proxyhost='vpnnode1',
            proxyuser='mpaproxy',
            identity='/home/mpaproxy/.ssh/mpr-id_rsa',
            user='perfact',
            down_script='myname',
            dryrun=True
        )
        expected = {
            'status': 0,
            'error': 0,
            'cmd': '',
            'msg': (
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 '
                'mpaproxy@vpnnode1 '
                'ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -i '
                '/home/mpaproxy/.ssh/mpr-id_rsa perfact@192.168.10.11 '
                '"sudo bash /etc/openvpn/down-cmd-tun && '
                'sudo bash /etc/openvpn/up-cmd-tun && '
                'sudo mv /etc/openvpn/myname /etc/openvpn/down-cmd-tun"'
            )
        }
        self.assertTrue(res == expected, msg=res)

        # Test with dangerous bash characters
        res = mpr_applyrules(
            ip='192.168.10.11 && dangerous',
            proxyhost='vpnnode1',
            proxyuser='mpaproxy',
            identity='/home/mpaproxy/.ssh/mpr-id_rsa',
            dryrun=True
        )
        expected = {
            'status': 1,
            'error': 0,
            'cmd': '',
            'msg': (
                'Forbidden characters for bash found in: '
                '192.168.10.11 && dangerous'
            )
        }
        self.assertTrue(res == expected, msg=res)
